import os
import pandas as pd
import sys
import numpy as np
import csv
from datetime import datetime

# Aktuelles Datum für Zeitstempelprüfung (wird nicht mehr benötigt, bleibt aber für Konsistenz)
CURRENT_DATE = datetime(2025, 3, 6)

# Liste der relevanten Parteien (außer für negative Schwankungen)
RELEVANT_PARTIES = {'CDU', 'CSU', 'SPD', 'AfD', 'FDP', 'BSW', 'FREIE WÄHLER', 'GRÜNE'}

# CSV-Datei für Anomalien
ANOMALIES_CSV = "anomalien.csv"

def get_next_id():
    """Ermittelt die nächste laufende Nummer basierend auf der bestehenden CSV-Datei."""
    if not os.path.exists(ANOMALIES_CSV):
        return 1
    with open(ANOMALIES_CSV, 'r', newline='') as f:
        reader = csv.reader(f)
        next(reader)  # Überspringe Header
        ids = [int(row[0]) for row in reader if row]
        return max(ids) + 1 if ids else 1

def append_to_csv(data):
    """Fügt eine Anomalie an die CSV-Datei an."""
    file_exists = os.path.exists(ANOMALIES_CSV)
    with open(ANOMALIES_CSV, 'a', newline='') as f:
        writer = csv.writer(f)
        if not file_exists:
            writer.writerow(["laufende_nummer", "partei", "bundesland", "stimmentyp", "zeit1", "zeit2", "anomalie_typ", "richtung", "beschreibung"])
        writer.writerow(data)

def extract_bundesland(file_path):
    """Extrahiert das Bundesland aus dem Dateinamen."""
    filename = os.path.basename(file_path)
    # Annahme: Dateiname ist im Format "data_kompakt_<bundesland>_<stimmentyp>.csv"
    if "data_kompakt_" in filename and (("_zweitstimmen" in filename) or ("_erststimmen" in filename)):
        parts = filename.split("data_kompakt_")[1].split("_")
        return parts[0]  # Das Bundesland ist der erste Teil nach "data_kompakt_"
    return "Unbekannt"  # Fallback, falls das Format nicht passt

def check_anomalies(file_path):
    """Überprüft eine kompakte Ergebnisdatei auf Anomalien und speichert sie in einer CSV-Datei, ignoriert ungültige Zeitstempel."""
    print(f"Analysiere Datei: {file_path}")
    
    # Extrahiere das Bundesland aus dem Dateinamen
    bundesland = extract_bundesland(file_path)
    
    try:
        df = pd.read_csv(file_path)
    except Exception as e:
        print(f"Fehler beim Lesen der Datei: {e}")
        return

    # Ausgabe der vorhandenen Spalten zur Diagnose
    print(f"Vorhandene Spalten: {list(df.columns)}")

    # Identifiziere Parteispalten
    standard_columns = ['timestamp', 'abgegebene_stimmen', 'ungültige_erststimmen', 'gültige_erststimmen', 
                       'ausgezählte_wahlkreise', 'ungültige_zweitstimmen', 'gültige_zweitstimmen']
    party_columns = [col for col in df.columns if col not in standard_columns]

    # Filtere nur relevante Parteien für die meisten Prüfungen
    relevant_party_columns = [col for col in party_columns if col.upper() in RELEVANT_PARTIES]

    # Bestimme, ob Erst- oder Zweitstimmen
    is_erststimmen = 'ungültige_erststimmen' in df.columns
    stimmentyp = "Erststimmen" if is_erststimmen else "Zweitstimmen"
    ungültige_col = 'ungültige_erststimmen' if is_erststimmen else 'ungültige_zweitstimmen'
    gültige_col = 'gültige_erststimmen' if is_erststimmen else 'gültige_zweitstimmen'

    # Prüfe, ob benötigte Spalten vorhanden sind
    if ungültige_col not in df.columns or gültige_col not in df.columns or 'abgegebene_stimmen' not in df.columns:
        print(f"Fehler: Erforderliche Spalten fehlen ({ungültige_col}, {gültige_col}, abgegebene_stimmen)")
        return

    ausgezählte_available = 'ausgezählte_wahlkreise' in df.columns
    if not ausgezählte_available:
        print("Warnung: Spalte 'ausgezählte_wahlkreise' fehlt, einige Prüfungen werden übersprungen.")
        max_ausgezählte = None
    else:
        max_ausgezählte = df['ausgezählte_wahlkreise'].max()

    anomalies_found = False
    anomaly_id = get_next_id()

    for index, row in df.iterrows():
        timestamp = str(row['timestamp'])
        abgegebene = row['abgegebene_stimmen']
        ungültige = row[ungültige_col]
        gültige = row[gültige_col]
        ausgezählte = row.get('ausgezählte_wahlkreise', 0) if ausgezählte_available else 0
        partei_summe = sum(row[col] for col in party_columns)

        # Auszählungsstand für diesen Zeitstempel
        ausgezählungsstand = (ausgezählte / max_ausgezählte * 100) if ausgezählte_available and max_ausgezählte > 0 else None
        ausgezählung_text = f", Auszählungsstand: {ausgezählungsstand:.1f}%" if ausgezählungsstand is not None else ""

        # 1. Inkonsistenz abgegebene vs. gültige + ungültige Stimmen (nicht in CSV)
        if abgegebene != gültige + ungültige:
            print(f"Anomalie bei {timestamp}: abgegebene_stimmen ({abgegebene}) != gültige ({gültige}) + ungültige ({ungültige}) in {bundesland}{ausgezählung_text}")
            anomalies_found = True

        # 2. Negative Werte (nur für relevante Parteien)
        if abgegebene < 0 or ungültige < 0 or gültige < 0 or (ausgezählte_available and ausgezählte < 0):
            desc = f"Negative Werte gefunden (abgegebene: {abgegebene}, ungültige: {ungültige}, gültige: {gültige}, ausgezählte: {ausgezählte})"
            print(f"Anomalie bei {timestamp}: {desc} in {bundesland}{ausgezählung_text}")
            append_to_csv([anomaly_id, "N/A", bundesland, stimmentyp, timestamp, timestamp, "Negative Werte", "negativ", desc])
            anomaly_id += 1
            anomalies_found = True
        for party in relevant_party_columns:
            if row[party] < 0:
                desc = f"Negative Stimmen für {party}: {row[party]}"
                print(f"Anomalie bei {timestamp}: {desc} in {bundesland}{ausgezählung_text}")
                append_to_csv([anomaly_id, party, bundesland, stimmentyp, timestamp, timestamp, "Negative Werte", "negativ", desc])
                anomaly_id += 1
                anomalies_found = True

        # 3. Ungewöhnlich hohe ungültige Stimmen (Schwellenwert: 2%)
        if abgegebene > 0 and (ungültige / abgegebene) > 0.02:
            desc = f"Hoher Anteil ungültiger Stimmen ({ungültige / abgegebene * 100:.1f}%)"
            print(f"Anomalie bei {timestamp}: {desc} in {bundesland}{ausgezählung_text}")
            append_to_csv([anomaly_id, "N/A", bundesland, stimmentyp, timestamp, timestamp, "Hohe ungültige Stimmen", "N/A", desc])
            anomaly_id += 1
            anomalies_found = True

        # 4. Parteistimmen vs. gültige Stimmen
        if partei_summe > gültige:
            desc = f"Summe der Parteistimmen ({partei_summe}) > gültige Stimmen ({gültige})"
            print(f"Anomalie bei {timestamp}: {desc} in {bundesland}{ausgezählung_text}")
            append_to_csv([anomaly_id, "N/A", bundesland, stimmentyp, timestamp, timestamp, "Parteistimmen > gültige", "N/A", desc])
            anomaly_id += 1
            anomalies_found = True
        elif partei_summe < gültige:
            desc = f"Summe der Parteistimmen ({partei_summe}) < gültige Stimmen ({gültige})"
            print(f"Anomalie bei {timestamp}: {desc} in {bundesland}{ausgezählung_text}")
            append_to_csv([anomaly_id, "N/A", bundesland, stimmentyp, timestamp, timestamp, "Parteistimmen < gültige", "N/A", desc])
            anomaly_id += 1
            anomalies_found = True

    # 6. Abnahme der ausgezählten Wahlkreise (nur wenn Spalte vorhanden)
    if ausgezählte_available and len(df) > 1:
        for i in range(1, len(df)):
            prev_ausgezählte = df['ausgezählte_wahlkreise'].iloc[i-1]
            curr_ausgezählte = df['ausgezählte_wahlkreise'].iloc[i]
            if curr_ausgezählte < prev_ausgezählte:
                curr_ausgezählungsstand = (curr_ausgezählte / max_ausgezählte * 100) if max_ausgezählte > 0 else 0
                desc = f"Abnahme der ausgezählten Wahlkreise ({prev_ausgezählte} -> {curr_ausgezählte})"
                print(f"Anomalie zwischen {df['timestamp'].iloc[i-1]} und {df['timestamp'].iloc[i]}: {desc} in {bundesland}, Auszählungsstand: {curr_ausgezählungsstand:.1f}%")
                append_to_csv([anomaly_id, "N/A", bundesland, stimmentyp, df['timestamp'].iloc[i-1], df['timestamp'].iloc[i], "Abnahme Wahlkreise", "negativ", desc])
                anomaly_id += 1
                anomalies_found = True

    # 7. Erweiterte Analyse: Extreme Schwankungen in Parteistimmen
    if len(df) > 1:
        changes = {party: df[party].diff().dropna() for party in party_columns}
        std_devs = {party: changes[party].std() for party in party_columns if len(changes[party]) > 0}

        for i in range(1, len(df)):
            prev_row = df.iloc[i-1]
            curr_row = df.iloc[i]
            prev_ts = prev_row['timestamp']
            curr_ts = curr_row['timestamp']
            prev_abgegebene = prev_row['abgegebene_stimmen']
            curr_abgegebene = curr_row['abgegebene_stimmen']
            prev_gültige = prev_row[gültige_col]
            curr_gültige = curr_row[gültige_col]
            ausgezählte_diff = curr_row['ausgezählte_wahlkreise'] - prev_row['ausgezählte_wahlkreise'] if ausgezählte_available else 0
            curr_ausgezählungsstand = (curr_row['ausgezählte_wahlkreise'] / max_ausgezählte * 100) if ausgezählte_available and max_ausgezählte > 0 else None
            ausgezählung_text = f", Auszählungsstand: {curr_ausgezählungsstand:.1f}%" if curr_ausgezählungsstand is not None else ""

            for party in party_columns:
                prev_votes = prev_row[party]
                curr_votes = curr_row[party]
                vote_diff = curr_votes - prev_votes

                # 7.1 Relative Schwankung (Schwellenwert: 200%, nur relevante Parteien)
                if party in relevant_party_columns and prev_votes > 0 and abs(vote_diff) / prev_votes > 2.0:
                    direction = "mehr als erwartet" if vote_diff > 0 else "weniger als erwartet"
                    desc = f"Relative Schwankung bei {party} ({prev_votes} -> {curr_votes}, {vote_diff / prev_votes * 100:.1f}%)"
                    print(f"Anomalie zwischen {prev_ts} und {curr_ts}: {desc} in {bundesland}, Richtung: {direction}{ausgezählung_text}")
                    append_to_csv([anomaly_id, party, bundesland, stimmentyp, prev_ts, curr_ts, "Relative Schwankung", "positiv" if vote_diff > 0 else "negativ", desc])
                    anomaly_id += 1
                    anomalies_found = True

                # 7.2 Kontextabhängige absolute Schwankung (nur relevante Parteien, mit Richtung)
                if party in relevant_party_columns and ausgezählte_available and ausgezählte_diff > 0 and prev_abgegebene > 0:
                    expected_change = (curr_abgegebene - prev_abgegebene) * (prev_votes / prev_abgegebene)
                    if abs(expected_change) >= 2000:
                        diff_from_expected = vote_diff - expected_change
                        if abs(diff_from_expected) / max(abs(expected_change), 1) > 1.0:
                            direction = "mehr als erwartet" if diff_from_expected > 0 else "weniger als erwartet"
                            desc = f"Unerwartete absolute Schwankung bei {party} (erwartet: {expected_change:.1f}, tatsächl.: {vote_diff})"
                            print(f"Anomalie zwischen {prev_ts} und {curr_ts}: {desc} in {bundesland}, Richtung: {direction}{ausgezählung_text}")
                            append_to_csv([anomaly_id, party, bundesland, stimmentyp, prev_ts, curr_ts, "Absolute Schwankung", "positiv" if diff_from_expected > 0 else "negativ", desc])
                            anomaly_id += 1
                            anomalies_found = True

                # 7.3 Sprung im Anteil an gültigen Stimmen (Schwellenwert: 2 Prozentpunkte, nur relevante Parteien)
                if party in relevant_party_columns and prev_gültige > 0 and curr_gültige > 0:
                    prev_share = prev_votes / prev_gültige * 100
                    curr_share = curr_votes / curr_gültige * 100
                    share_diff = curr_share - prev_share
                    if abs(share_diff) > 2.0:
                        direction = "mehr als erwartet" if share_diff > 0 else "weniger als erwartet"
                        desc = f"Anteilssprung bei {party} (springt von {prev_share:.1f}% auf {curr_share:.1f}%)"
                        print(f"Anomalie zwischen {prev_ts} und {curr_ts}: {desc} in {bundesland}, Richtung: {direction}{ausgezählung_text}")
                        append_to_csv([anomaly_id, party, bundesland, stimmentyp, prev_ts, curr_ts, "Anteilssprung", "positiv" if share_diff > 0 else "negativ", desc])
                        anomaly_id += 1
                        anomalies_found = True

                # 7.4 Statistische Analyse mit Sigma-Wert (Schwellenwert: 3 Sigma, nur relevante Parteien)
                if party in relevant_party_columns and party in std_devs and std_devs[party] > 0 and abs(vote_diff) > 3 * std_devs[party]:
                    sigma = abs(vote_diff) / std_devs[party]
                    direction = "positiv" if vote_diff > 0 else "negativ"
                    desc = f"Statistische Anomalie bei {party} (Änderung {vote_diff}, Sigma: {sigma:.2f}, StdDev: {std_devs[party]:.1f})"
                    print(f"Anomalie zwischen {prev_ts} und {curr_ts}: {desc} in {bundesland}, Richtung: {direction}{ausgezählung_text}")
                    append_to_csv([anomaly_id, party, bundesland, stimmentyp, prev_ts, curr_ts, "Statistische Anomalie", direction, desc])
                    anomaly_id += 1
                    anomalies_found = True

                # 7.5 Negative Schwankung (für alle Parteien)
                if vote_diff < 0:
                    desc = f"Negative Schwankung bei {party} ({prev_votes} -> {curr_votes})"
                    print(f"Anomalie zwischen {prev_ts} und {curr_ts}: {desc} in {bundesland}, Richtung: negativ{ausgezählung_text}")
                    append_to_csv([anomaly_id, party, bundesland, stimmentyp, prev_ts, curr_ts, "Negative Schwankung", "negativ", desc])
                    anomaly_id += 1
                    anomalies_found = True

                # 7.6 Korrelation mit ausgezählten Wahlkreisen (nur relevante Parteien, mit Richtung)
                if party in relevant_party_columns and ausgezählte_available and ausgezählte_diff > 0 and prev_votes > 0:
                    expected_change = (curr_abgegebene - prev_abgegebene) * (prev_votes / prev_abgegebene)
                    if abs(expected_change) > 0:
                        diff_from_expected = vote_diff - expected_change
                        if abs(diff_from_expected) / abs(expected_change) > 1.0:
                            direction = "mehr als erwartet" if diff_from_expected > 0 else "weniger als erwartet"
                            desc = f"Unerwartete Schwankung bei {party} (erwartet: {expected_change:.1f}, tatsächl.: {vote_diff})"
                            print(f"Anomalie zwischen {prev_ts} und {curr_ts}: {desc} in {bundesland}, Richtung: {direction}{ausgezählung_text}")
                            append_to_csv([anomaly_id, party, bundesland, stimmentyp, prev_ts, curr_ts, "Unerwartete Schwankung", "positiv" if diff_from_expected > 0 else "negativ", desc])
                            anomaly_id += 1
                            anomalies_found = True

    # 8. Abweichungen von der Wahlbeteiligung
    if len(df) > 1:
        for i in range(1, len(df)):
            prev_abgegebene = df['abgegebene_stimmen'].iloc[i-1]
            curr_abgegebene = df['abgegebene_stimmen'].iloc[i]
            curr_ausgezählungsstand = (df['ausgezählte_wahlkreise'].iloc[i] / max_ausgezählte * 100) if ausgezählte_available and max_ausgezählte > 0 else None
            if prev_abgegebene > 0 and curr_abgegebene == 0:
                ausgezählung_text = f", Auszählungsstand: {curr_ausgezählungsstand:.1f}%" if curr_ausgezählungsstand is not None else ""
                desc = f"Abgegebene Stimmen fallen auf 0 (vorher: {prev_abgegebene})"
                print(f"Anomalie bei {df['timestamp'].iloc[i]}: {desc} in {bundesland}{ausgezählung_text}")
                append_to_csv([anomaly_id, "N/A", bundesland, stimmentyp, df['timestamp'].iloc[i-1], df['timestamp'].iloc[i], "Abweichung Wahlbeteiligung", "negativ", desc])
                anomaly_id += 1
                anomalies_found = True

    if not anomalies_found:
        print("Keine Anomalien gefunden.")

def main():
    """Hauptfunktion zur Anomalieprüfung mit Kommandozeilenparameter."""
    if len(sys.argv) != 2:
        print("Verwendung: python script.py <pfad_zu_kompakt_datei>")
        print("Beispiel: python script.py data_kompakt_bayern_zweitstimmen.csv")
        sys.exit(1)

    file_path = sys.argv[1]
    if not os.path.isfile(file_path):
        print(f"Fehler: Datei '{file_path}' existiert nicht.")
        sys.exit(1)

    check_anomalies(file_path)

if __name__ == "__main__":
    main()
