import os
import hashlib
import argparse
import time
from collections import defaultdict

def get_file_hash(filepath, hash_algorithm='sha256'):
    """Berechnet den Hash einer Datei."""
    hash_func = hashlib.new(hash_algorithm)
    with open(filepath, 'rb') as f:
        for chunk in iter(lambda: f.read(4096), b""):
            hash_func.update(chunk)
    return hash_func.hexdigest()

def find_duplicate_files(directory):
    """Findet doppelte Dateien anhand des Hashwerts."""
    hash_dict = defaultdict(list)
    
    for root, _, files in os.walk(directory):
        for file in files:
            filepath = os.path.join(root, file)
            try:
                file_hash = get_file_hash(filepath)
                hash_dict[file_hash].append((filepath, os.path.getmtime(filepath)))
            except Exception as e:
                print(f"Fehler beim Verarbeiten von {filepath}: {e}")
    
    return hash_dict

def clean_duplicates(directory, dry_run=True):
    """Löscht doppelte Dateien, behält aber die älteste und die jüngste Datei."""
    hash_dict = find_duplicate_files(directory)
    files_to_delete = []
    
    for file_list in hash_dict.values():
        if len(file_list) > 2:
            file_list.sort(key=lambda x: x[1])  # Sortieren nach Änderungszeit
            to_delete = file_list[1:-1]  # Alle außer älteste und jüngste
            files_to_delete.extend([f[0] for f in to_delete])
    
    for file in files_to_delete:
        if dry_run:
            print(f"Würde löschen: {file}")
        else:
            try:
                os.remove(file)
                print(f"Gelöscht: {file}")
            except Exception as e:
                print(f"Fehler beim Löschen von {file}: {e}")
    
    print(f"{len(files_to_delete)} Dateien zum Löschen gefunden.")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Doppelte Dateien in einem Verzeichnis bereinigen.")
    parser.add_argument("directory", help="Pfad zum Verzeichnis, das bereinigt werden soll")
    parser.add_argument("--delete", action="store_true", help="Tatsächlich löschen anstatt nur anzuzeigen")
    
    args = parser.parse_args()
    
    clean_duplicates(args.directory, dry_run=not args.delete)
